/* batch_Baseball-RunScored-1.jsx
   - Saves under .../{league}/
   - Each run writes: LEAGUE_teamId_ABBR_<RUN_TAG>-<runNum><ANIM_TAIL>/00000.png
   - Replaces RunNumPng with 1.png..4.png
   - Replaces ScoreTextPng with "RUN SCORED.png" (run 1) or "RUNS SCORED.png" (run 2–4)
   - Applies Tint on ExpandingLayer (Map Black/White To = Primary; Amount=100)
*/
(function () {
  // ---------- utils ----------
  function env(k,d){ var v=$.getenv(k); return (v===null||v===undefined||v==="")?d:v; }
  function trim(s){ var t=String(s||""); while(/^[ \t\r\n]/.test(t)) t=t.substring(1); while(/[ \t\r\n]$/.test(t)) t=t.substring(0,t.length-1); return t; }
  function toLower(s){ return String(s||"").toLowerCase(); }
  function clamp01(v){ return Math.max(0, Math.min(1, v)); }
  function sanitize(s){ var bad='\\/:*?"<>|', t=String(s||""), out=""; for(var i=0;i<t.length;i++){ var ch=t.charAt(i); out+=(bad.indexOf(ch)>=0?"-":ch);} return out; }
  function numOr(v,def){ var n=parseFloat(v); return (isFinite(n)?n:def); }
  function safeColor(c){ if(!c||c.length!==3)return[0,0,0]; function cl(x){return(x<0?0:(x>1?1:x));} return [cl(numOr(c[0],0)),cl(numOr(c[1],0)),cl(numOr(c[2],0))]; }
  function rgb01(r,g,b){ return [clamp01(numOr(r,0)/255), clamp01(numOr(g,0)/255), clamp01(numOr(b,0)/255)]; }
  function isBlackish(c){ return (!c || c.length!==3 || (c[0]<=0.05 && c[1]<=0.05 && c[2]<=0.05)); }
  function lightGrey(){ return [0.7,0.7,0.7]; }
  function smartPrimary(p,s){ return isBlackish(p) ? (isBlackish(s)?lightGrey():s) : p; }
  function smartSecondary(p,s){ return isBlackish(s) ? (isBlackish(p)?lightGrey():p) : s; }

  // ---------- env ----------
  var PROJECT   = env("AE_PROJECT", null);
  var CSV_PATH  = env("AE_CSV", null);
  var COMP_NAME = env("AE_COMP","RunScore");

  var TEAMNAME_LAYER      = env("AE_TEAMNAME_LAYER","TeamName");
  var TEAMLOGO_LAYER      = env("AE_TEAMLOGO_LAYER","TeamLogo");
  var EXPANDINGLINE_LAYER = env("AE_EXPANDINGLINE_LAYER","ExpandingLayer");

  // NEW: PNG layers + dir
  var RUNNUM_PNG_LAYER    = env("AE_RUNNUM_PNG_LAYER","RunNumPng");
  var SCORETEXT_PNG_LAYER = env("AE_SCORETEXT_PNG_LAYER","RunScoredPng");
  var TEXT_PNG_DIR        = env("AE_TEXT_PNG_DIR","");

  // Optional: HomePlate variants if used in the comp
  var HOMEPLATE_LAYER = env("AE_HOMEPLATE_LAYER","HomePlate-1.png");
  var HOMEPLATE_PATH  = env("AE_HOMEPLATE_PATH","");

  var LOGO_DIR  = env("AE_LOGO_DIR","");
  var LOGO_EXTS = env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd").split(",");

  var LEAGUE    = env("AE_LEAGUE","MLB");
  var TEAMS_LIST= env("AE_TEAMS","");
  var LIMIT_STR = env("AE_LIMIT","");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = env("AE_OUTDIR","");
  var PATH_TPL  = env("AE_PATH_TEMPLATE","{league}"); // league-only
  var RS_TPL    = env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL    = env("AE_OM_TEMPLATE","PNG Sequence");

  var RUN_TAG   = env("AE_RUN_TAG","R");
  var ANIM_TAIL = env("AE_ANIM_TAIL","_1_3");
  function normTail(s){ s=String(s||""); if(!s) return ""; if(s.charAt(0)!=="_") s="_"+s; return sanitize(s); }

  var PURGE     = (env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER = (env("AE_NO_RENDER","0")==="1");
  var QUIT_APP  = (env("AE_QUIT","1")==="1");

  // ---------- CSV ----------
  function openRead(path){ var f=new File(path); if(!f.exists) fail("File not found: "+path); f.open("r"); var s=f.read(); f.close(); return s; }
  function fail(msg){ alert(msg); throw new Error(msg); }
  function parseCSV(txt){
    var raw=txt.split(/\r\n|\n|\r/), lines=[], i; for(i=0;i<raw.length;i++){ var L=raw[i]; if(L && !/^\s*$/.test(L)) lines.push(L); }
    var rows=[], c; for(i=0;i<lines.length;i++){ var line=lines[i]; var cells=(line.indexOf("\t")!==-1? line.split("\t") : line.split(",")); 
      for(c=0;c<cells.length;c++){ var cell=trim(cells[c]); if(cell.charAt(0)=='"' && cell.charAt(cell.length-1)=='"') cell=cell.substring(1,cell.length-1); cells[c]=cell; }
      rows.push(cells);
    }
    if(rows.length<2) fail("CSV has no data."); return rows;
  }
  function headerIdx(h){
    var m={}, i; for(i=0;i<h.length;i++) m[toLower(h[i])]=i;
    function need(x){ if(m[x]===undefined) fail("Missing column: "+x); }
    need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
    return m;
  }

  function isWhiteish(c){
  return (c && c.length===3 && c[0]>=0.98 && c[1]>=0.98 && c[2]>=0.98);
}

  function buildTeams(rows){
    var h=rows[0], idx=headerIdx(h), out=[], i;
    var leagueIdx = idx["league"];
    var nameIdx   = (idx["name"]!==undefined ? idx["name"] : idx["displayname"]);
    var tidIdx    = (idx["espn_team_id"]!==undefined ? idx["espn_team_id"]
                   : (idx["team_id"]!==undefined ? idx["team_id"]
                   : (idx["id"]!==undefined ? idx["id"] : undefined)));
    for(i=1;i<rows.length;i++){
      var r=rows[i]; var ab=trim(r[idx["abbreviation"]]); if(!ab) continue;
      out.push({
        abbr: ab,
        league: (leagueIdx!==undefined ? trim(r[leagueIdx]) : "MLB"),
        name:   (nameIdx!==undefined   ? trim(r[nameIdx])   : ab),
        team_id: (tidIdx!==undefined   ? trim(r[tidIdx])    : ""),
        primary:   rgb01(r[idx["r"]],  r[idx["g"]],  r[idx["b"]]),
        secondary: rgb01(r[idx["r2"]], r[idx["g2"]], r[idx["b2"]])
      });
    }
    return out;
  }
  function pickTeams(all){
    var res=[], i;
    if (TEAMS_LIST){ var want={}, parts=TEAMS_LIST.split(","); for(i=0;i<parts.length;i++){ want[trim(parts[i])] = true; } for(i=0;i<all.length;i++){ if(want[all[i].abbr]) res.push(all[i]); } }
    else if (LEAGUE){ for(i=0;i<all.length;i++){ if(toLower(all[i].league)===toLower(LEAGUE)) res.push(all[i]); } }
    else { res = all.slice(0); }
    if (LIMIT && res.length>LIMIT) res = res.slice(0, LIMIT);
    return res;
  }

  // ---------- AE helpers ----------
  function findComp(name){ for(var i=1;i<=app.project.numItems;i++){ var it=app.project.item(i); if(it instanceof CompItem && it.name===name) return it; } return null; }
  function getLayer(comp, name){ try{ return comp.layer(name); }catch(e){ return null; } }
  function getLayerAlt(comp, primary, alts){
    var L=null; try{ L=comp.layer(primary);}catch(e){}
    if (L) return L;
    alts=alts||[];
    for (var i=0;i<alts.length;i++){ try{ L=comp.layer(alts[i]); }catch(e){} if (L) return L; }
    return null;
  }

function setTextContentNoStyle(layer, text){
  var st = layer && layer.property("Source Text"); if(!st) return false;
  var td = st.value; td.text = String(text||"");
  try { st.setValue(td); return true; } catch(e){ return false; }
}

function setTextFillOnly(layer, color){
  var st = layer && layer.property("Source Text"); if(!st) return false;
  var td = st.value;
  td.applyFill = true;
  td.fillColor = safeColor(color);
  // DO NOT touch stroke flags/color/width
  try { st.setValue(td); return true; } catch(e){ return false; }
}

// Find a text layer by name, searching into precomps (depth-limited)
function findTextLayerDeep(comp, name, depth){
  depth = depth||0; if (depth > 3) return null;
  try {
    var L = comp.layer(name);
    if (L && L.matchName === "ADBE TextLayer") return L;
  } catch(e){}
  for (var i=1; i<=comp.numLayers; i++){
    var child = comp.layer(i);
    try {
      if (child && child.source instanceof CompItem){
        var hit = findTextLayerDeep(child.source, name, depth+1);
        if (hit) return hit;
      }
    }catch(e){}
  }
  return null;
}


  function replaceFootage(comp, layerName, filePath){
    var ly = getLayerAlt(comp, layerName, []);
    if (!ly) { $.writeln("[PNG] Layer not found: "+layerName); return false; }
    var f = new File(filePath); if (!f.exists) { $.writeln("[PNG] Missing file: "+f.fsName); return false; }
    try{
      var io = new ImportOptions(f);
      if (!io.canImportAs(ImportAsType.FOOTAGE)) return false;
      var footage = app.project.importFile(io);
      ly.replaceSource(footage, false);
      return true;
    }catch(e){ $.writeln("[PNG] Replace failed on "+layerName+": "+e); return false; }
  }

  // Tint on ExpandingLayer (locale-proof)
  function setTintEffectColors(layer, color){
    if (!layer) return false;
    var fx = layer.property("ADBE Effect Parade") || layer.property("Effects"); if (!fx) return false;
    var tint=null;
    for (var i=1;i<=fx.numProperties;i++){ var e=fx.property(i); if (e && e.matchName==="ADBE Tint"){ tint=e; break; } }
    if (!tint){ try{ tint = fx.addProperty("ADBE Tint"); }catch(e){ tint=null; } }
    if (!tint) { $.writeln("[Tint] Could not add Tint on '"+layer.name+"'"); return false; }
    try{ tint.enabled=true; }catch(e){}
    var c=safeColor(color);
    for (var j=1;j<=tint.numProperties;j++){
      var p=tint.property(j);
      try{
        if (p.propertyValueType===PropertyValueType.COLOR) p.setValue(c);
        if (p.propertyValueType===PropertyValueType.ONE_D) p.setValue(100);
      }catch(_){}
    }
    try{ tint.moveTo(fx.numProperties); }catch(e){}
    $.writeln("[Tint] Applied to '"+layer.name+"'");
    return true;
  }

  // Optional HomePlate switcher
  function replaceHomePlate(comp, layerName, runNum){
    if (!HOMEPLATE_PATH) return;
    var lyr=null; try{ lyr=comp.layer(layerName);}catch(e){}
    if (!lyr){
      for (var i=1;i<=comp.numLayers;i++){ var t=comp.layer(i); if (t && t.name && t.name.indexOf("HomePlate")===0){ lyr=t; break; } }
    }
    if (!lyr) return;
    var f = new File(HOMEPLATE_PATH + "/HomePlate-" + runNum + ".png"); if (!f.exists) return;
    var io=new ImportOptions(f); if (!io.canImportAs(ImportAsType.FOOTAGE)) return;
    var footage=app.project.importFile(io); lyr.replaceSource(footage,false);
    try{ lyr.name = "HomePlate-" + runNum + ".png"; }catch(e){}
  }

  // ---------- run ----------
  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("MLB RunScored - PNG Sequences");

  if(!PROJECT) fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) fail("AE_CSV env not set.");
  var rows=parseCSV(openRead(CSV_PATH)), teams=buildTeams(rows), todo=pickTeams(teams);
  if(!todo.length) fail("No teams matched.");

  var comp=findComp(COMP_NAME); if(!comp) fail("Comp not found: "+COMP_NAME);

  // output root
  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  if(!rootOut.exists) rootOut.create();

  // per team
  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    var leagueLabel = t.league || LEAGUE || "MLB";
    var teamIdSafe  = sanitize(t.team_id || "NA");
    var abbrSafe    = sanitize(t.abbr);

    var P = smartPrimary( safeColor(t.primary),   safeColor(t.secondary) );
    var S = smartSecondary( safeColor(t.primary), safeColor(t.secondary) );

    // Output subfolder: .../{league}/
    var sub = PATH_TPL.replace("{league}", sanitize(leagueLabel)).replace("{abbr}", abbrSafe);
    var outFolder = new Folder(rootOut.fsName + "/" + sub); if (!outFolder.exists) outFolder.create();

    for (var runNum=1; runNum<=4; runNum++){
      // Logo swap
      if (LOGO_DIR){
        var logoFile=null;
        for (var li=0; li<LOGO_EXTS.length; li++){
          var test = new File(LOGO_DIR + "/" + t.abbr + "." + trim(LOGO_EXTS[li]));
          if (test.exists){ logoFile = test; break; }
        }
        if (logoFile){
          try{
            var io=new ImportOptions(logoFile);
            if (io.canImportAs(ImportAsType.FOOTAGE)){
              var footage=app.project.importFile(io);
              var ly=comp.layer(TEAMLOGO_LAYER); if (ly) ly.replaceSource(footage,false);
            }
          }catch(e){}
        }
      }

    // 2) TeamName — update text and fill (keep stroke unchanged)
    var nameLy = getLayer(comp, TEAMNAME_LAYER) || findTextLayerDeep(comp, TEAMNAME_LAYER, 0);
    if (nameLy){
      var teamText = (t.name || t.abbr || "").toUpperCase();
      var fill = P;                        // smartPrimary from your code
      if (isWhiteish(fill)) fill = lightGrey();

      setTextContentNoStyle(nameLy, teamText); // change ONLY the text string
      setTextFillOnly(nameLy, fill);           // change ONLY the fill; leave stroke alone
    } else {
      $.writeln("[TeamName] layer not found: " + TEAMNAME_LAYER);
    }


      // --- NEW: replace the number PNG ---
      if (TEXT_PNG_DIR){
        replaceFootage(comp, RUNNUM_PNG_LAYER, TEXT_PNG_DIR + "/" + runNum + ".png");

        // --- NEW: replace the "RUN(S) SCORED" PNG ---
        var scoreName = (runNum===1 ? "RUN SCORED.png" : "RUNS SCORED.png");
        replaceFootage(comp, SCORETEXT_PNG_LAYER, TEXT_PNG_DIR + "/" + scoreName);
      }

      // Optional: HomePlate variants
      replaceHomePlate(comp, HOMEPLATE_LAYER, runNum);

      // Tint ExpandingLayer
      var expLy = getLayerAlt(comp, EXPANDINGLINE_LAYER, ["ExpandingLine"]);
      if (expLy){ setTintEffectColors(expLy, P); }

      if (PURGE && app.purge){ try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

      if (!NO_RENDER){
        var animName = sanitize(leagueLabel) + "_" + teamIdSafe + "_" + abbrSafe +
                       "_" + sanitize(RUN_TAG) + "-" + runNum + normTail(ANIM_TAIL);

        var animFolder = new Folder(outFolder.fsName + "/" + animName); if (!animFolder.exists) animFolder.create();
        var outFile = File(animFolder.fsName + "/" + animName);

        var rqItem = app.project.renderQueue.items.add(comp);
        try{ rqItem.applyTemplate(RS_TPL); }catch(e){}
        var om = rqItem.outputModule(1);

        // robust PNG OM
        (function applyPNGOMOrDie(om, preferred){
          var want=[]; if(preferred) want.push(preferred);
          want.push("PNG Sequence","PNG Sequence with Alpha","PNG","PNG-sequence");
          var avail=[]; try{ avail=om.templates||[]; }catch(e){}
          function cieq(a,b){ return String(a).toLowerCase()===String(b).toLowerCase(); }
          for (var i1=0;i1<want.length;i1++){ for (var j1=0;j1<avail.length;j1++){ if (cieq(avail[j1],want[i1])){ try{ om.applyTemplate(avail[j1]); return; }catch(e){} } } }
          for (var k1=0;k1<avail.length;k1++){ if (String(avail[k1]).toLowerCase().indexOf("png")!==-1){ try{ om.applyTemplate(avail[k1]); return; }catch(e){} } }
          throw new Error("No PNG Output Module template found. Available: ["+(avail.join(", ")||"none")+"]");
        })(om, OM_TPL);

        om.file = outFile;
        app.project.renderQueue.render();
        while(app.project.renderQueue.numItems>0){ app.project.renderQueue.item(1).remove(); }
      }
    }
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
  if (QUIT_APP) app.quit();

})();
